/*:
 * @target MZ
 * @plugindesc v1.1 Localize SimpleVoice "Voice Volume" label via variable with safe title fallback
 * @author HS
 *
 * @help
 * SimpleVoiceのオプション項目「ボイス音量」のラベル文字列を多言語化します。
 * ゲーム中は指定した変数の文字列をラベルとして使用し、タイトル画面など
 * 変数が未初期化の場面では各言語のフォールバック文字列を表示します。
 *
 * 【使い方】
 * 1) プラグイン順: HS_LangSimple → (任意のブリッジ) → SimpleVoice → 本プラグイン
 * 2) 本プラグインの labelVarId に「ラベル用の変数ID」を設定（例: 171）
 * 3) ゲーム中にその変数へ各言語の文字列をセット（例: "ボイス 音量" / "Voice Volume"）
 * 4) タイトルから開いた場合は fb0..fb3 のフォールバックが使われます
 *
 * ※ HS_LangSimple の ConfigManager.hs_langIndex を参照して言語indexを判断します。
 *
 * @param labelVarId
 * @text ラベル用 変数ID
 * @type variable
 * @default 0
 * @desc ゲーム中、この変数に入っている「文字列」をオプション名として表示します（例: 171）。
 *
 * @param fb0
 * @text 0:日本語
 * @default ボイス 音量
 *
 * @param fb1
 * @text 1:English
 * @default Voice Volume
 *
 * @param fb2
 * @text 2:简体中文
 * @default 语音音量
 *
 * @param fb3
 * @text 3:한국어
 * @default 보이스 음량
 */
(() => {
  const PN = "HS_SV_OptionLocalize";
  const P  = PluginManager.parameters(PN);
  const VAR = Number(P.labelVarId || 0);
  const FB = [P.fb0 || "", P.fb1 || "", P.fb2 || "", P.fb3 || ""];

  // HS_LangSimple が保存する現在の言語index（0:JP, 1:EN, ...）
  function langIndex() {
    return Number(ConfigManager && ConfigManager.hs_langIndex != null
      ? ConfigManager.hs_langIndex : 0);
  }

  function isNonEmptyString(v) {
    return typeof v === "string" && v.trim() !== "";
  }

  // 表示ラベル決定：ゲーム中は変数の「文字列」だけ採用（数値0は無視）／タイトル等はフォールバック
  function labelText() {
    if (VAR > 0 && window.$gameVariables && $gameVariables._data) {
      const v = $gameVariables.value(VAR);
      if (isNonEmptyString(v)) return String(v);
    }
    const idx = langIndex();
    const fb = FB[idx] || FB[0];
    return isNonEmptyString(fb) ? fb : "Voice Volume";
  }

  function relabel(win) {
    if (!win || !win._list) return;
    for (let i = 0; i < win._list.length; i++) {
      const it = win._list[i];
      if (it && it.symbol === "voiceVolume") {
        it.name = labelText();
      }
    }
  }

  // SimpleVoiceが音量項目を追加した直後に差し替える
  const _addVolumeOptions = Window_Options.prototype.addVolumeOptions;
  Window_Options.prototype.addVolumeOptions = function() {
    _addVolumeOptions.apply(this, arguments);
    try { relabel(this); } catch (e) { /* no-op */ }
  };

  // 言語切替後の再描画でも反映されるように
  const _refresh = Window_Options.prototype.refresh;
  Window_Options.prototype.refresh = function() {
    try { relabel(this); } catch (e) { /* no-op */ }
    _refresh.call(this);
  };
})();

